/*
Candidate Number - J1918

30th May 1997

Object Oriented Programming Assignment - Animated Page Editor
=============================================================

The main design of this program has been targetted towards making it as neat
as possible to add (and remove) new types of items to the current collection of
three.  (Note - "item" is used here to denote a component which can be drawn on
the pages (eg text, rectangles etc).  This is to avoid confusing with the
"component" class used in Java).  There are two main classes which are specific
to each type of item.  The first is the "item handler" and the other is the
"item" itself.  In addition, there can be as many classes as need be which can
be instantiated from either of these classes.

The general way in which the classes communicate with each other is as follows:


    editor (1 instance)
        ^
        |
        v
screenHandler (1 instance) <-> itemHandler (1 instance per item type)
        ^
   /    |    \
  v     v     v
 pageHandler (3 (= number of pages) instances)
  ^     ^     ^
  |     |    /|\\\
  |     |    vvvvvv
  |    /|\\\ item (1 instance per item on this page)
  |    vvvvvv
 /|\\\ item (1 instance per item on this page)
 vvvvvv
 item (1 instance per item on this page)


Each item may then communicate with any other classes specific to that type of
item.  For example, the bouncing ball item communicates with instances of the
class "bouncingBallItemThread" which are threads that draw the balls.

The "editor" does all the general stuff in the main applet, outside the screen
which shows the different pages.  So, buttons, text fields, and the screen are
all created here.  The "screenHandler" handles everything that goes on inside
the display (the visible canvas) that is not specific to a given page.  So, for
example, mouse clicks are acknowledged here and then passed on to the current
item handler which then decides how to react to them.

The "item handler" is responsible for handling everything to do with adding
the item to the page.  For example, after the user has clicked on the button
for this item and then clicks on the page, the "boolean mouseDown(Point p1)"
method in the item handler is called.  Similarly, when the mouse key is
released, the mouseUp(Point p2) method is called.  These are the only events
which are ever used to draw items on a page and so are, initially, the only
methods called.  This idea means that future item types are able to be created
following numerous mouse clicks (not just one or two).

However, if, after one of these events, the item handler has received
sufficient information to be able to try and create a new item, then the whole
process of adding an item begins.  Before returning "true" (to indicate this
state of affairs) the item handler creates a new Rectangle to represent the
area of the page that it wishes to cover.  The pageHandler recieves this
rectangle and uses it to check whether it can accept an item, of this type, on
to the page.  It does this by first checking whether the rectangle interferes
with existing items and then by asking the item handler whether it is happy to
add the item (eg rectangles with no area may not be worth adding).  If both of
these conditions are true then the item is created by the item handler and then
initiated by the pageHandler using the item's "init()" method.

The "itemHandler" class is an abstract class which is extended to a subclass
for each type of item (eg the rectangle item extends this class to
rectangleItemHandler).  The bouncing ball handler is actually a subclass of
rectangleItemHandler because both items are created so similarly.

That is the entire job of each item handler.  It serves no purpose in drawing,
removing etc the items on the page.  That is left to the "item" class which is
extended for each type of item (eg the rectangle item extends this class to
rectangleItem).  The bouncing ball "item" is actually a subclass of
rectangleItem, mainly because its handler is the corresponding subclass.  After
an item has been initialised, it waits around for the pageHandler to tell it
to either "show" itself, "hide" itself, or "kill" itself.  The item will then
handle everything else (like threads) all by itself.  In the end it was decided
that it was better not to get each item to hide itself by drawing over itself
in the background colour, but instead to let the pageHandler wipe over the
entire page.  This is much faster, but less object oriented.  The line or two
of code that would implement the more OO way, have been commented out.

See diagram for the class hierachy.

As an example of what would need to be done to add a new subclass of item:

If we wished to add, say, a new type of item which  shows an animation
of a fireworks display in a rectangular box, then all we would need to do is
create a new class (the "item handler) to handle the setting up of each item,
a new class (the "item" itself) to handle how the item behaves while it is a
member of the page, and any threads which need to be created.  The only changes
that need to be made to the existing code, is to add a new button to the
main applet, and to update the "screen handler" so that a new instance for the
item handler is created.  Also, when the button is pressed, the screen needs
to be told that this item handler is the next item handler that is going to be
used.

Here is a brief description of all the public methods:

public class editor extends Applet
  public void init() - initialises the buttons, text field and screen handler
  public boolean action(Event e, Object arg) - tell screenHandler that a
                                               button has been pressed

class screenHandler extends Canvas
  screenHandler(editor e) - creates a new pageHandler for each page
  public void buttonPressed(Object arg) - handle the pressing of one of the
                                          buttons
  public boolean mouseDown(Event evt, int x, int y) - handle the pressing down
                                                      of the mouse key
  public boolean mouseUp(Event evt, int x, int y)   - handle the releasing
                                                      of the mouse key
  public String getText() - return the contents of the text field ("input")
  public void paint(Graphics g) - tell page to draw all its items
  public void status(String s) - update applet's status display

class pageHandler
  pageHandler(screenHandler sh) - simply stores the screenHandler "sh" pointer
  public void deleteAll() - delete all elements from current page
  public void addItem(ItemHandler ih) - Add the item if possible
  public void deleteItem(Point p) - delete item below Point p, if one exists
  public void show() - show all items on page
  public void hide() - hide all items on page
  public int getSize() - return the number of items on the page

abstract class ItemHandler
  ItemHandler(screenHandler sh) - simply stores the screenHandler "sh" pointer
  public boolean mouseDown(Point p1) - how the item handles the pressing down
                                       of a mouse key given mouse coord p1
  public boolean mouseUp(Point p2)   - how the item handles the releasing
                                       of a mouse key given mouse coord p2
  abstract protected void createRectangle() -this will define the command to
                                             create a Rectangle from the inputs
  public Rectangle getRectangle()
  abstract public Item createItem() - this will create a new instance of the
                                      item to be placed on the page
  public boolean checkMe() - check to make sure that item handler is happy with
                             the inputs it has been given to try and create an
			     item with

abstract class Item
  Item(screenHandler sh, Rectangle rect) - simply stores parameters in private
                                           variables
  public void init() - carried out as soon as the item is to be first placed
                       on the page
  public void kill() - remove the item from the page by drawing over it in
                       back colour
  abstract public synchronized void show() - this will show the item on the
                                             page
  public Rectangle getRectangle() - return what rectangle is represented by
                                    this item


The other classes are all subclasses of Item or ItemHandler and so there is
no point in describing their public methods here.  A detailed desription of
how they work is given within the source code.

I would say that the most important classes in this program are Item and
ItemHandler as it is these classes which enable new types of items to be
added to the program.

Now for the source code...

*/

import java.applet.*;
import java.awt.*;
import java.util.*;

public class editor extends Applet
{
  private Panel button = new Panel(); // panel for buttons
  private screenHandler sh = new screenHandler(this); // screen for items
  public static TextField input = new TextField(); // to enter strings

  public void init()
  {
    resize(800,600);

    // buttons
    button.add(new Button("Next Page"));
    button.add(new Button("Delete Item"));
    button.add(new Button("Delete All"));
    button.add(new Button("Text"));
    button.add(new Button("Rectangle"));
    button.add(new Button("Bouncing Ball"));
   
    setLayout(new BorderLayout());

    // add and position components
    add("North", button);
    add("Center", sh);
    add("South", input);
  }

  public boolean action(Event e, Object arg)
  {
    // tell screenHandler that a button has been pressed
    sh.buttonPressed(arg);
    return true;
  }
}

class screenHandler extends Canvas
{
  // ** screenHandler handles a screen! **

  private int number_of_pages = 3; // set total number of pages
  private int page_number = 0; // start off on the first page
  private editor e; // nice to know the editor since this has "input" and
                    // "showStatus()".

  //private String next = new String(""); // next item to be added to page
  private Point mouse = new Point(-1,-1); // position of last mouseDown()
  // deleting is true iff we are in the mode where clicking on page deletes!
  private boolean deleting = false; // default to false

  // this array points to the objects (pageHandler) to handle each page
  private pageHandler[] ph = new pageHandler[number_of_pages];
  // item handlers
  private ItemHandler texth = new textItemHandler(this);
  private ItemHandler rectangleh = new rectangleItemHandler(this);
  private ItemHandler bouncingBallh = new bouncingBallItemHandler(this);
  // item handler for next item to be added (default to text handler)
  private ItemHandler nextItemHandler = texth;

  screenHandler(editor e)
  {
    this.e = e;
    // create a new pageHandler for each page
    for (int i = 0; i < number_of_pages; i++)
      ph[i] = new pageHandler(this);
  }

  public void buttonPressed(Object arg)
  {
    // ** handle the pressing of one of the buttons **

    // no longer deleting
    deleting = false;
    // tell screenHandler about button clicks
    if ("Next Page".equals(arg))
    {
      // ** turn over to next page **
      // tell current page to hide
      ph[page_number].hide();
      
      // change to the next page
      page_number = (page_number + 1) % number_of_pages;
      
      // tell current page to show itself
      ph[page_number].show();

      status("Page turned!");
    }
    else if ("Delete Item".equals(arg))
      // next action to be applied to page is to delete an item
      deleting = true;
    else if ("Delete All".equals(arg))
    {
      // tell page to delete all its items
      ph[page_number].deleteAll();
    }
    else if ("Text".equals(arg))
      // change which item handler will be used for the next addition to page
      nextItemHandler = texth;
    else if ("Rectangle".equals(arg))
      // change which item handler will be used for the next addition to page
      nextItemHandler = rectangleh;
    else if ("Bouncing Ball".equals(arg))
      // change which item handler will be used for the next addition to page
      nextItemHandler = bouncingBallh;
  }

  public boolean mouseDown(Event evt, int x, int y)
  {
    // ** handle the pressing down of the mouse key **

    mouse = new Point(x,y); // get "down" coordinate

    if (deleting)
      // straight away, ask pageHandler to try to delete any item below mouse
      ph[page_number].deleteItem(mouse);
    else if (nextItemHandler.mouseDown(mouse))
      // item handler is ready to add, so ask page to try to add item to itself
      ph[page_number].addItem(nextItemHandler);
    return true;
  }

  public boolean mouseUp(Event evt, int x, int y)
  {
    // ** handle the releasing of the mouse key **

    mouse = new Point(x,y); // get "up" coordinate

    if (deleting)
      // nothing to do
      {}
    else if (nextItemHandler.mouseUp(mouse))
      // item handler is ready to add, so ask page to try to add item
      ph[page_number].addItem(nextItemHandler);
    return true;
  }

  public String getText()
  {
    // ** return the contents of the text field ("input") **
    return (e.input).getText();
  }

  public void paint(Graphics g)
  {
    // ** tell page to draw all its items **
    ph[page_number].show();
  }

  public void status(String s)
  {
    // ** update applet's status display **
    e.showStatus(s + " - Page " + (page_number + 1) +
		   " has " + ph[page_number].getSize() + " items.");
  }
}

class pageHandler
{
  // ** pageHandler handles a page! **

  private screenHandler sh;
  private Vector items = new Vector(); // vector holding all the items
  // private Item newItem; // used for any item being added

  pageHandler(screenHandler sh)
  {
    this.sh = sh; // the screenHandler that is looking after this page
  }

  public void deleteAll()
  {
    // ** delete all elements from current page **
    while (items.size() > 0)
    {
      // deal with the item at the head of the vector
      ((Item) items.elementAt(0)).kill(); // tell item to kill
      items.removeElementAt(0); // remove from main vector
    }

    sh.status("Page cleared!");
  }

  public void addItem(ItemHandler ih)
  {
    // ** Add the item if possible **

    // get object representing area we are trying to add to
    Rectangle rect = ih.getRectangle();

    // test whether new item will collide with an existing item
    boolean checkedOthers = true; // safe to add it (default)
    for (int i = 0; i < items.size(); i++)
      // cycling through each existing item
      if ((((Item) items.elementAt(i)).rect).intersects(rect))
        checkedOthers = false;

    // only add item if all items are happy about it
    if (checkedOthers && ih.checkMe())
    {
      // item is safe to be added to page - it wont interfere with anything
      Item newItem = ih.createItem(); // create a new item
      items.addElement(newItem); // add to vector of items for this page
      newItem.init(); // initialise item on the screen

      sh.status("Item added!");
    }
    else
    {
      sh.status("Item not added!");
    }
  }

  public void deleteItem(Point p)
  {
    // ** delete item below Point p, if one exists **

    // check to see whether point lies inside any existing rectangles
    for (int i = 0; i < items.size(); i++)
      if ((((Item) items.elementAt(i)).getRectangle()).inside(p.x, p.y))
      {
	// just clicked inside an existing item - so hide it and then delete it
	((Item) items.elementAt(i)).kill();
	items.removeElementAt(i); // remove from main vector
	sh.status("Item deleted!");
	break; // no more to be done
      }
  }

  public void show()
  {
    // ** show all items on page **

    // cycle through the items, asking each one to show itself
    for (int i = 0; i < items.size(); i++)
      ((Item) (items.elementAt(i))).show();
  }

  public void hide()
  {
    // ** hide all items on page **

    // tell items they are about to be hidden (ie threads must be handled)
    for (int i = 0; i < items.size(); i++)
      ((Item) (items.elementAt(i))).hide();

    // wipe over entire screen to blank it - this is much faster than telling
    //   each individual item to wipe itself
    (sh.getGraphics()).clearRect(0,0,(sh.size()).width,(sh.size()).height);
  }

  public int getSize()
  {
    // ** return the number of items on the page **
    return items.size();
  }
}

abstract class ItemHandler
{
  protected screenHandler sh;
  protected Point p1, p2;
  protected Rectangle rect;

  ItemHandler(screenHandler sh)
  {
    this.sh = sh;
  }

  public boolean mouseDown(Point p)
  {
    // ** how the item handles the pressing down of a mouse key **
    p1 = p;
    return false;
  }

  public boolean mouseUp(Point p)
  {
    // ** how the item handles the releasing of a mouse key **
    p2 = p;
    return false;
  }

  // ** this will define the command to create a Rectangle from the inputs **
  abstract protected void createRectangle();

  public Rectangle getRectangle()
  {
    // ** return the rectangle representing the area to be covered **
    return rect;
  }

  // ** this will create a new instance of the item to be placed on the page **
  abstract public Item createItem();

  public boolean checkMe()
  {
    // ** check to make sure that item handler is happy with the inputs it **
    // ** has been given to try and create an item with **

    // by default, dont bother with items which cover no area
    return !(rect.isEmpty());
  }
}

abstract class Item
{
  protected Rectangle rect;
  protected screenHandler sh;
  protected Graphics g;

  Item(screenHandler sh, Rectangle rect)
  {
    this.sh = sh;
    g = sh.getGraphics();
    this.rect = rect;
  }

  public void init()
  {
    // ** carried out as soon as the item is to be first placed on the page **
    show(); // show item
  }

  public void kill()
  {
    // ** remove the item from the page by drawing over it in back colour **
    g.clearRect(rect.x, rect.y, rect.width, rect.height);    
  }

  // ** this will show the item on the page **
  abstract public synchronized void show();

  public void hide()
  {
    // ** hide item from the screen **

    // no need to remove itself as the pageHandler will do this for everyone
    // However, its still good to tell the item that it is about to be removed
    // so that items with threads can handle those threads if necessary

    // old code to paint over the old item with the background colour
    //g.clearRect(rect.x, rect.y, rect.width, rect.height);
  }

  public Rectangle getRectangle()
  {
    // ** return what rectangle is represented by this item **
    return rect;
  }
}

class textItemHandler extends ItemHandler
{
  private String str;
  private FontMetrics font;
  private int h;

  textItemHandler(screenHandler sh)
  {
    super(sh);
  }

  public boolean mouseDown(Point p)
  {
    p1 = p;
    font = (sh.getGraphics()).getFontMetrics(); // font details
    h = font.getHeight(); // font height
    str = sh.getText(); // text stored in text field ("input")
    createRectangle();
    return true; // tell the screen handler that we have all the inputs needed
  }

  protected void createRectangle()
  {
    // create rectangle based on constructors parameters and font details
    rect = new Rectangle(p1.x, p1.y - h, font.stringWidth(str), h);
  }

  public Item createItem()
  {
    // try to add the text item to the page as soon as mouse is pressed
    return new textItem(sh, str, rect); // create new item and return it
  }
}

class textItem extends Item
{
  private String str;
  private int offset;

  textItem(screenHandler sh, String str, Rectangle rect)
  {
    super(sh, rect);
    this.str = str; // string to be shown
    offset = (g.getFontMetrics()).getAscent(); // y offset within rectangle
    show(); // show the item when first placed on page
  }

  public void show()
  {
    // draw the actual string (which should all appear above and to the right)
    g.setColor(Color.black);
    g.drawString(str, rect.x, rect.y + offset);
  }
}

class rectangleItemHandler extends ItemHandler
{
  rectangleItemHandler(screenHandler sh)
  {
    super(sh);
  }

  public boolean mouseUp(Point p)
  {
    p2 = p;
    createRectangle();
    return true; // tell the screen handler that we have all the inputs needed
  }

  protected void createRectangle()
  {
    // create rectangle based on constructors parameters and font details
    rect = new Rectangle
      (Math.min(p1.x, p2.x), Math.min(p1.y, p2.y), // make sure p1 is top-left
       Math.abs(p2.x - p1.x), Math.abs(p2.y - p1.y)); // coord, and width and
                                                      // height are positive
  }

  public Item createItem()
  {
    // create a new rectangle item to add to the page and return it
    return new rectangleItem(sh, rect);
  }
}

class rectangleItem extends Item
{
  rectangleItem(screenHandler sh, Rectangle rect)
  { 
    super(sh, rect);
  }

  public void show()
  {
    // draw the actual rectangle on the page
    g.setColor(Color.blue);
    g.fillRect(rect.x, rect.y, rect.width, rect.height);
  }
}

class bouncingBallItemHandler extends rectangleItemHandler
{
  bouncingBallItemHandler(screenHandler sh)
  {
    super(sh);
  }

  public void createRectangle()
  {
    // create rectangle as we did with rectangle item
    super.createRectangle();
    // make sure that rectangle is not too small for the ball
    rect.resize(Math.max(rect.width, 12), Math.max(rect.height, 12));
  }

  public Item createItem()
  {
    // create a new rectangle item to add to the page and return it
    return new bouncingBallItem(sh, rect);
  }
}

class bouncingBallItem extends rectangleItem
{
  private bouncingBallItemThread ball1, ball2;

  bouncingBallItem(screenHandler sh, Rectangle rect)
  {
    super(sh, rect);
  }

  public void init()
  {
    // best to make new balls now so that we are ready to start drawing
    ball1 = new bouncingBallItemThread(sh, rect);
    ball2 = new bouncingBallItemThread(sh, rect);
    // show the item
    show();
    // start both balls
    ball1.start();
    ball2.start();
  }

  public void kill()
  {
    // kill the threads
    ball1.stop();
    ball2.stop();
    // now do the usual stuff - ie wipe over the item
    super.kill();
  }

  public void show()
  {
    // draw the actual rectangle on the page
    g.setColor(Color.white);
    g.drawRect(rect.x, rect.y, rect.width - 1, rect.height - 1);
    // show the balls last
    ball1.show();
    ball2.show();
  }

  public void hide()
  {
    // hide both balls
    ball1.hide();
    ball2.hide();
  }
}

class bouncingBallItemThread extends Thread
{
  private Rectangle rect;
  private screenHandler sh;
  private Graphics g;
  private int x, y; // starting coords of ball relative to item
  // randomize direction to start moving in
  private int dx = 4 * (int) (Math.rint(Math.random()) + 0.5) - 2;
  private int dy = 4 * (int) (Math.rint(Math.random()) + 0.5) - 2;
  private boolean visible = true; // true iff the ball is to be drawn

  bouncingBallItemThread(screenHandler sh, Rectangle rect)
  {
    this.sh = sh;
    g = sh.getGraphics();
    this.rect = rect;
    // randomize starting position
    x = (int) Math.rint((rect.width - 12) * Math.random()) + 1;
    y = (int) Math.rint((rect.height - 12) * Math.random()) + 1;
    show();
  }

  public synchronized void show()
  {
    draw(Color.red); // draw red ball
    visible = true; // make sure ball is actually shown
  }

  public synchronized void hide()
  {
    // no need to hide the ball since the pageHandler will do it for me
    //draw(sh.getBackground());
    visible = false; // make sure ball is not shown
  }

  public void run()
  {
    try { sleep(100); }
    catch (InterruptedException e) {}
    while (true)
    {
      move(); // move ball
      try { sleep(20); }
      catch (InterruptedException e) {}
    }
  }

  private synchronized void draw(Color c)
  {
    Color oldColor = g.getColor(); // remember current colour so can restore it
    g.setColor(c); // set new colour for ball
    g.fillOval(x + rect.x, y + rect.y, 10, 10); // draw ball
    g.setColor(oldColor); // restore old colour
  }

  private synchronized void move()
  {
    if (visible) // only draw when ball is supposed to be visible
      draw(sh.getBackground()); // "hide" ball

    // the next 5 lines control the movement of the ball
    // (no reason why balls should stop moving just because they cant be seen)
    x = x + dx; y = y + dy;
    if (x < 1) { x = 1; dx = -dx; }
    if (x + 10 >= rect.width - 1) { x = rect.width - 11; dx = -dx; }
    if (y < 1) { y = 1; dy = -dy; }
    if (y + 10 >= rect.height - 1) { y = rect.height - 11; dy = -dy; }

    if (visible) // only draw when ball is supposed to be visible
      draw(Color.red); // draw red ball
  }
}

/*

The End

Candidate Number - J1918

*/

